package de.serra.graph_walker;

import de.serra.graph_walker.GraphWalkerStackFrame.State;

import org.checkerframework.checker.nullness.qual.Nullable;
import org.checkerframework.dataflow.qual.Pure;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Arrays;

/**
 * Entry point to start the traversal of an object graph.
 * <p>
 * Methods in this class can be overriden to customize the traversal.
 *
 * @author Peter Lamby
 */
public class GraphWalk {
	private final ObjectgraphVisitor visitor;
	private final @Nullable GraphWalk parentWalk;
	private Control control = Control.CONTINUE;
	private @Nullable GraphWalkerStackFrame<?>[] stack = new GraphWalkerStackFrame[16];
	private int stackSize;
	private VisitController controller;

	/**
	 * Constructs.
	 *
	 * @param visitor The visitor to use for this GraphWalk.
	 */
	public GraphWalk(final ObjectgraphVisitor visitor) {
		this(visitor, null);
	}

	/**
	 * Constructs a walk. When {@code parentWalk} is non {@code null} the walk is a subwalk.
	 *
	 * @param visitor    The visitor to use for this GraphWalk.
	 * @param parentWalk The optional parent walk that used to detect cycles.
	 */
	private GraphWalk(final ObjectgraphVisitor visitor, final @Nullable GraphWalk parentWalk) {
		this.visitor = visitor;
		this.parentWalk = parentWalk;
		controller = new GraphWalkVisitController();
	}

	/**
	 * Start the traversal of {@code toWalk}.
	 *
	 * @param toWalk The root of the object graph.
	 */
	public void walk(final @Nullable Object toWalk) {

		if (toWalk == null) {
			visitor.visitNull(controller);
			return;
		}

		stack[0] = new GraphWalkerStackFrame<Object>(State.BEFORE_OBJECT, toWalk);
		stackSize++;

		dispatch();

		// reset for next walk;
		control = Control.CONTINUE;
		stack[0] = new GraphWalkerStackFrame<Object>(State.BEFORE_OBJECT, toWalk);
		stackSize = 0;
		for (int i = 1; i < stack.length; i++) {
			stack[i] = null;
		}
		controller = new GraphWalkVisitController();
	}

	/**
	 * Creates a "sub" graphwalk from the current position.
	 * <p>
	 * It remembers the already visited nodes but otherwise acts as a walk on it's own.
	 *
	 * @param visitor The visitor to use for this GraphWalk.
	 * @return the new graph walk.
	 */
	public GraphWalk createSubwalk(final ObjectgraphVisitor visitor) {
		return new GraphWalk(visitor, this);
	}

	/**
	 * Determins which method to call depending on the state of the current
	 * {@link de.serra.graph_walker.GraphWalkerStackFrame StackFrame}.
	 */
	protected void dispatch() {
		while (!isDone()) {
			final var stackFrame = currentStackFrame();
			switch (stackFrame.state()) {
			case BEFORE_OBJECT:
				beforeObject();
				break;
			case ARRAY:
				visitArray();
				break;
			case ARRAY_MEMBER:
				visitArrayMember();
				break;
			case CLASS:
				visitClass();
				break;
			case CLASS_FIELD:
				visitClassField();
				break;
			default:
				break;
			}
		}
	}

	/**
	 * Calls {@link ObjectgraphVisitor#beforeObject(Object, VisitController, boolean)} and determins the next state
	 * depending on the object.
	 */
	protected void beforeObject() {
		final var stackFrame = currentStackFrame();

		if (stackFrame.isLeaving()) {
			leave();
			return;
		}

		final var value = stackFrame.value();
		var alreadySeen = value != null && alreadySeen(value);
		if (parentWalk != null) {
			alreadySeen |= value != null && parentWalk.alreadySeen(value);
			alreadySeen |= parentWalk.currentStackFrame().value() == value;
		}
		visitor.beforeObject(value, controller, alreadySeen);

		if (isStopped()) {
			return;
		}
		if (alreadySeen || isDontGoDeeper()) {
			leave();
			return;
		}

		if (value == null) {
			visitor.visitNull(controller);
			leave();
			return;
		}

		final var type = value.getClass();
		if (type.isArray()) {
			pushOnStack(new GraphWalkerStackFrame<Object>(State.ARRAY, value));
			return;
		}

		pushOnStack(new GraphWalkerClassState(value));
	}

	/**
	 * Visits an array.
	 */
	protected void visitArray() {
		final var stackFrame = currentStackFrame();

		if (stackFrame.isLeaving()) {
			leave();
			return;
		}

		assert isDontGoDeeper() == false : "isDontGoDeeper in visitArray";

		final var array = stackFrame.value();
		assert array != null : "visitArray should never be called with null array";

		final var arrayType = array.getClass();
		final var componentType = arrayType.getComponentType();

		if (componentType == byte.class) {
			visitor.visitByteArray((byte[]) array, controller);
			leave();
		} else if (componentType == short.class) {
			visitor.visitShortArray((short[]) array, controller);
			leave();
		} else if (componentType == int.class) {
			visitor.visitIntArray((int[]) array, controller);
			leave();
		} else if (componentType == long.class) {
			visitor.visitLongArray((long[]) array, controller);
			leave();
		} else if (componentType == float.class) {
			visitor.visitFloatArray((float[]) array, controller);
			leave();
		} else if (componentType == double.class) {
			visitor.visitDoubleArray((double[]) array, controller);
			leave();
		} else if (componentType == boolean.class) {
			visitor.visitBooleanArray((boolean[]) array, controller);
			leave();
		} else if (componentType == char.class) {
			visitor.visitCharArray((char[]) array, controller);
			leave();
		} else {
			final Object[] objectArray = (Object[]) array;
			visitor.visitArray(objectArray, controller);

			if (!isDontGoDeeper() && objectArray.length > 0) {
				pushOnStack(new GraphWalkerArrayMemberState(objectArray, 0));
			} else {
				if (!isStopped()) {
					visitor.leaveArray(objectArray, controller);
				}
				leave();
				return;
			}
		}
	}

	/**
	 * Visits an array member.
	 */
	protected void visitArrayMember() {
		final GraphWalkerArrayMemberState stackFrame = (GraphWalkerArrayMemberState) currentStackFrame();

		if (!stackFrame.isLeaving()) {
			assert !isDontGoDeeper() : "isDontGoDeeper in visitArrayMember";

			visitor.visitArrayMember(stackFrame.member(), stackFrame.idx(), stackFrame.value(), controller);

			if (isStopped()) {
				return;
			}

			if (!isDontGoDeeper()) {
				// continue traversing the current member
				pushOnStack(new GraphWalkerStackFrame<@Nullable Object>(State.BEFORE_OBJECT, stackFrame.member()));
				return;
			} else {
				// we are done with this arrayMember
				// try to go to the next one or leave
			}
		} else {
			// coming back from a deeper member visit
			// we can continue with the next element
			// but must not visit the current array member again
		}

		visitor.leaveArrayMember(stackFrame.member(), stackFrame.idx(), stackFrame.value(), controller);
		if (isStopped()) {
			return;
		}

		// try to go to the next array member

		// Since we visit a brand new member we clear the control.
		resetControl();

		if (stackFrame.tryGoNextMember()) {
			return;
		}

		visitor.leaveArray(stackFrame.value(), controller);
		leave();
	}

	/**
	 * Visits an "normal" object.
	 */
	protected void visitClass() {
		final GraphWalkerClassState stackFrame = (GraphWalkerClassState) currentStackFrame();

		if (!stackFrame.fieldsAlreadyVisited()) {
			if (!stackFrame.isUpwardsInClassHierarchy()) {
				visitor.visitClass(stackFrame.value(), controller);
			} else {
				visitor.visitSuperClass(stackFrame.value(), stackFrame.getPositionInClassHierarchy(), controller);
			}
		}

		if (isStopped()) {
			return;
		}

		if (!isDontGoDeeper() && !stackFrame.fieldsAlreadyVisited()) {
			// visit fields
			final var fields = stackFrame.getPositionInClassHierarchy().getDeclaredFields();
			if (fields.length > 0) {
				// there are fields to visit
				pushOnStack(new GraphWalkerClassFieldState(stackFrame.value(), fields));
				return;
			}
		}

		if (!isDontGoSuper() && stackFrame.tryGoUpInClassHierarchy()) {
			// not at the top yet
			// Just leave. We will immediatly be called again
			// with the position in the class hierarchy set to one step
			// higher due to tryGoUpInClassHierarchy.
			return;
		}

		visitor.leaveClass(stackFrame.value(), controller);
		leave();
	}

	/**
	 * Visits the field of an Object.
	 */
	protected void visitClassField() {
		final GraphWalkerClassFieldState stackFrame = (GraphWalkerClassFieldState) currentStackFrame();

		if (!stackFrame.isLeaving()) {
			assert !isDontGoDeeper() : "isDontGoDeeper in visitClassField";

			visitor.visitClassField(stackFrame.value(), stackFrame.getField(), controller);

			if (isStopped()) {
				return;
			}

			if (!isDontGoDeeper()) {
				// visit the value of the field
				final var field = stackFrame.getField();
				final var instance = stackFrame.value();

				boolean shouldVisit = true;
				// don't visit if primitive
				shouldVisit &= !field.getType().isPrimitive();

				if (shouldVisit) {
					final boolean canAccess;
					if (Modifier.isStatic(field.getModifiers())) {
						canAccess = canAccessStaticField(field);
					} else {
						canAccess = field.canAccess(instance);
					}
					if (!canAccess) {
						// don't visit if we can't access the field value
						shouldVisit &= field.trySetAccessible();
						// NOTE should we log this?
					}
				}

				if (shouldVisit) {
					try {
						final Object fieldValue = field.get(instance);
						pushOnStack(new GraphWalkerStackFrame<@Nullable Object>(State.BEFORE_OBJECT, fieldValue));
						return;
					} catch (IllegalArgumentException | IllegalAccessException e) {
						// NOTE should we log this?
					}
				}
				// we could not visit the field
			}
		} else {
			// coming back from a field visit.
		}

		visitor.leaveClassField(stackFrame.value(), stackFrame.getField(), controller);
		if (isStopped()) {
			return;
		}

		// try to go to the next field
		// set state to continue
		resetControl();

		if (stackFrame.tryGoNextField()) {
			return;
		}

		// NOTE we could probably just reuse the leave state to notify visitClass that
		// we are finished with all fields
		goBack();
		final GraphWalkerClassState classState = (GraphWalkerClassState) currentStackFrame();
		classState.fieldsDone();
	}

	@SuppressWarnings("nullness")
	private boolean canAccessStaticField(final Field f) {
		assert Modifier.isStatic(f.getModifiers()) : "Field is not static";
		return f.canAccess(null);
	}

	/**
	 * Used to determin if the current traversal is done.
	 * <p>
	 * Either because we {@link Control#STOP stopped} or because there is nothing more to visit.
	 *
	 * @return {@code true} if the traversal is done.
	 */
	@Pure
	protected boolean isDone() {
		return stackSize == 0 || control == Control.STOP;
	}

	/**
	 * Is the traversal stopped?
	 *
	 * @return {@code true} if the traversal is stopped.
	 */
	@Pure
	protected boolean isStopped() {
		return control == Control.STOP;
	}

	/**
	 * Should the children not be traversed.
	 *
	 * @return {@code true} if the children should not be traversed.
	 */
	@Pure
	protected boolean isDontGoDeeper() {
		return control == Control.STOP || control == Control.CONTINUE_BUT_DONT_GO_DEEPER;
	}

	/**
	 * Should the parent classes not be traversed.
	 *
	 * @return {@code true} if the parent classes should not be traversed.
	 */
	@Pure
	protected boolean isDontGoSuper() {
		return control == Control.STOP || control == Control.CONTINUE_BUT_DONT_GO_SUPER;
	}

	/**
	 * Returns the current stack frame.
	 *
	 * @return the current stack frame.
	 */
	@Pure
	protected GraphWalkerStackFrame<?> currentStackFrame() {
		return relativeFromCurrent(0);
	}

	/**
	 * Returns a stack frame relative to the current one.
	 *
	 * @param offset The offset. {@code 0} to get the current one.
	 * @return the specified stack frame.
	 */
	@Pure
	protected GraphWalkerStackFrame<?> relativeFromCurrent(final int offset) {
		assert stackSize > 0 + offset : "Illegal State. No more element in stack";
		final var ret = stack[stackSize - (offset + 1)];
		assert ret != null : "Illegal State. Trying to access non existing stack element";
		return ret;
	}

	/**
	 * Removes the current stack frame and informs the next stack frame that we are returning to it.
	 */
	protected void leave() {
		goBack();
		if (!isDone()) {
			final @Nullable GraphWalkerStackFrame<?> state = currentStackFrame();
			if (state != null) {
				state.leave();
			}

		}
	}

	/**
	 * Used to detect circular object graphs.
	 * <p>
	 * Should return {@code true} if {@code o} was already visited in the current path.
	 * <p>
	 * If this returns {@code true} traversal will skip {@code o}.
	 *
	 * @param o The object to check.
	 * @return {@code true} if we already visited {@code o} in the current path.
	 */
	@Pure
	protected boolean alreadySeen(final Object o) {
		for (int i = stackSize - 2; i >= 0; i--) {
			assert stack[i] != null;

			if (stack[i].state() == State.BEFORE_OBJECT && stack[i].value() == o) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Adds a new stack frame on top of the stack.
	 *
	 * @param statestackFrame The stack frame to add.
	 */
	protected void pushOnStack(final GraphWalkerStackFrame<?> statestackFrame) {
		if (stackSize == stack.length) {
			grow(stackSize + 1);
		}
		stack[stackSize] = statestackFrame;
		stackSize++;
	}

	/**
	 * Increases the capacity to ensure that it can hold at least the number of elements specified by the minimum
	 * capacity argument.
	 *
	 * @param minCapacity the desired minimum capacity
	 */
	protected void grow(final int minCapacity) {
		final int oldCapacity = stack.length;
		final int newCapacity = Math.max(
				// grow by half of current size.
				oldCapacity + oldCapacity >> 1, minCapacity);
		// realloc
		stack = Arrays.copyOf(stack, newCapacity);
	}

	/**
	 * Used to continue the traversal as normal. Used after leaving an object to reset any flags only meant for that
	 * node.
	 */
	protected void resetControl() {
		assert control != Control.STOP : "Should never try to continue after stop.";

		this.control = Control.CONTINUE;
	}

	/**
	 * Removes the current stack frame.
	 */
	protected void goBack() {
		stackSize--;
		stack[stackSize] = null;
	}

	private class GraphWalkVisitController extends VisitController {
		@Override
		public void stop() {
			control = Control.STOP;
		}

		@Override
		public void dontGoDeeper() {
			if (control == Control.STOP) {
				throw new IllegalStateException("Cant set dontGoDeeper if already stopped");
			}
			control = Control.CONTINUE_BUT_DONT_GO_DEEPER;
		}

		@Override
		public boolean isStopped() {
			return control == Control.STOP;
		}

		@Override
		public GraphWalk createSubWalk(ObjectgraphVisitor visitor) {
			return GraphWalk.this.createSubwalk(visitor);
		}
	}

	/**
	 * Controls the traversal.
	 *
	 */
	public enum Control {
		/**
		 * Immediatly stop the traversal.
		 */
		STOP,
		/**
		 * The default. Continue the traversal as usual.
		 */
		CONTINUE,
		/**
		 * Don't visit any members or fields of the curent object.
		 */
		CONTINUE_BUT_DONT_GO_DEEPER,
		/**
		 * Don't visit any (more) super classes of the current object.
		 */
		CONTINUE_BUT_DONT_GO_SUPER;
	}
}
