package de.serra.graph_walker;

import de.serra.graph_walker.GraphWalkerState.State;

import org.checkerframework.checker.nullness.qual.Nullable;

import java.lang.reflect.Modifier;

/**
 * Entry point to start the traversal.
 *
 * @author Peter Lamby
 * @see #visit(Object, ObjectgraphVisitor)
 */
public final class GraphWalker {
	private GraphWalker() {
	}

	/**
	 * Visit {@code toVisit} depth-first.
	 * <p>
	 * While traversing the object graph {@code visitor} is called back to inform it about the the traversal.
	 *
	 * @param toVisit The object to traverse.
	 * @param visitor The visitor.
	 */
	public static void visit(final Object toVisit, final ObjectgraphVisitor visitor) {
		var ctx = new GraphWalkerContext(toVisit, visitor);

		if (toVisit == null) {
			visitor.visitNull(ctx.getController());
			return;
		}

		visit(ctx);
	}

	private static void visit(final GraphWalkerContext ctx) {
		while (!ctx.isDone()) {
			var current = ctx.current();
			assert current != null : "When we are not done current() should have a state";
			switch (current.state) {
			case OBJECT:
				dispatchObject(ctx);
				break;
			case ARRAY:
				visitArray(ctx);
				break;
			case ARRAY_MEMBER:
				visitArrayMember(ctx);
				break;
			case CLASS:
				visitClass(ctx);
				break;
			case CLASS_FIELD:
				visitClassField(ctx);
				break;
			default:
				assert false : "Illegal State. The state is not known";
			}
		}
	}

	private static void dispatchObject(final GraphWalkerContext ctx) {
		var state = ctx.current();
		assert state != null : "dispatchObject should never be called with null state";

		if (state.isLeaving()) {
			ctx.leave();
			return;
		}

		var value = state.value;
		var alreadySeen = value != null && ctx.alreadySeen(value);
		ctx.visitor.beforeObject(value, ctx.getController(), alreadySeen);

		if (ctx.isStopped()) {
			return;
		}
		if (alreadySeen || ctx.isDontGoDeeper()) {
			ctx.leave();
			return;
		}

		if (value == null) {
			ctx.visitor.visitNull(ctx.getController());
			ctx.leave();
			return;
		}

		var type = value.getClass();
		if (type.isArray()) {
			ctx.pushOnStack(value, State.ARRAY);
			return;
		}

		ctx.pushOnStack(new GraphWalkerClassState(value));
	}

	private static void visitArray(final GraphWalkerContext ctx) {
		assert ctx.current() != null : "visitArray should never be called with null state";

		// we are done with this entire array.
		if (ctx.current().isLeaving()) {
			ctx.leave();
			return;
		}

		assert ctx.isDontGoDeeper() == false : "isDontGoDeeper in visitArray. This doesn't make sense";

		var array = ctx.current().value;
		assert array != null : "visitArray should never be called with null array";

		var arrayType = array.getClass();
		var componentType = arrayType.getComponentType();

		if (componentType == byte.class) {
			ctx.visitor.visitByteArray((byte[]) array, ctx.getController());
			ctx.leave();
		} else if (componentType == short.class) {
			ctx.visitor.visitShortArray((short[]) array, ctx.getController());
			ctx.leave();
		} else if (componentType == int.class) {
			ctx.visitor.visitIntArray((int[]) array, ctx.getController());
			ctx.leave();
		} else if (componentType == long.class) {
			ctx.visitor.visitLongArray((long[]) array, ctx.getController());
			ctx.leave();
		} else if (componentType == float.class) {
			ctx.visitor.visitFloatArray((float[]) array, ctx.getController());
			ctx.leave();
		} else if (componentType == double.class) {
			ctx.visitor.visitDoubleArray((double[]) array, ctx.getController());
			ctx.leave();
		} else if (componentType == boolean.class) {
			ctx.visitor.visitBooleanArray((boolean[]) array, ctx.getController());
			ctx.leave();
		} else if (componentType == char.class) {
			ctx.visitor.visitCharArray((char[]) array, ctx.getController());
			ctx.leave();
		} else {
			Object[] objectArray = (Object[]) array;

			ctx.visitor.visitArray(objectArray, ctx.getController());

			if (!ctx.isDontGoDeeper() && objectArray.length > 0) {
				ctx.pushOnStack(new GraphWalkerArrayMemberState(objectArray[0], 0));
			} else {
				if (!ctx.isStopped()) {
					ctx.visitor.leaveArray(objectArray, ctx.getController());
				}
				ctx.leave();
				return;
			}
		}
	}

	private static void visitArrayMember(final GraphWalkerContext ctx) {
		GraphWalkerArrayMemberState memberState = (GraphWalkerArrayMemberState) ctx.current();
		assert memberState != null : "visitArrayMember should never be called with null state";

		assert ctx.relativeFromCurrent(1) != null : "visitArrayMember should have a direct parent";
		assert ctx.relativeFromCurrent(1).value instanceof Object[]
				: "visitArrayMember should have Object[] as direct parent";

		@Nullable
		Object[] array = (@Nullable Object[]) ctx.relativeFromCurrent(1).value;

		assert array != null : "Illegal state. The array itself can't be null when visiting one of its members.";

		if (!memberState.isLeaving()) {
			// we are visiting the member for the first time

			assert !ctx.isDontGoDeeper() : "Illegal state";

			ctx.visitor.visitArrayMember(memberState.value, memberState.idx, array, ctx.getController());

			if (ctx.isStopped()) {
				return;
			}

			if (!ctx.isDontGoDeeper()) {
				// continue traversing the current member
				ctx.pushOnStack(memberState.value, State.OBJECT);
				return;
			} else {
				// we are done with this arrayMember
				// try to go to the next one or leave
			}
		} else {
			// coming back from a deeper member visit
			// we can continue with the next element
			// but must not visit the array member again
		}

		// try to go to the next array member

		// set state to continue
		ctx.resetControl();

		var nextMemberIdx = memberState.idx + 1;
		if (nextMemberIdx < array.length) {
			// go to next array member
			ctx.replaceCurrent(new GraphWalkerArrayMemberState(array[nextMemberIdx], nextMemberIdx));
		} else {
			// no more array members
			ctx.visitor.leaveArray(array, ctx.getController());
			ctx.leave();
		}
	}

	private static void visitClass(final GraphWalkerContext ctx) {
		GraphWalkerClassState state = (GraphWalkerClassState) ctx.current();
		// NOTE maybe support visiting statics in the future. In that case the instance
		// CAN be null
		assert state != null : "Illegal state. The instance can not be null when visiting its class";
		assert !ctx.isDontGoDeeper() : "isDontGoDeeper set. Should have been caught by dispatchObject";

		if (!state.fieldsAlreadyVisited()) {
			if (!state.isUpwardsInClassHierarchy()) {
				ctx.visitor.visitClass(state.value, ctx.getController());
			} else {
				ctx.visitor.visitSuperClass(state.value, state.getPositionInClassHierarchy(), ctx.getController());
			}

		}

		if (ctx.isStopped()) {
			return;
		}

		if (!ctx.isDontGoDeeper() && !state.fieldsAlreadyVisited()) {
			// visit fields
			var fields = state.getPositionInClassHierarchy().getDeclaredFields();
			if (fields.length > 0) {
				// there are fields to visit
				ctx.pushOnStack(new GraphWalkerClassFieldState(state.value, fields));
				return;
			}
		}

		if (!ctx.isDontGoSuper()) {
			if (state.tryGoUpInClassHierarchy()) {
				// not at the top yet
				return;
			}
		}

		// all fields and superclasses are visited
		ctx.visitor.leaveClass(state.value, ctx.getController());
		ctx.leave();
	}

	private static void visitClassField(final GraphWalkerContext ctx) {
		GraphWalkerClassFieldState state = (GraphWalkerClassFieldState) ctx.current();
		assert state != null : "Illegal state. The instance can not be null when visiting its fields";

		if (!state.isLeaving()) {
			// visit the current field
			assert !ctx.isDontGoDeeper() : "Illegal state";
			assert !ctx.isStopped() : "Illegal state";

			ctx.visitor.visitClassField(state.value, state.getField(), ctx.getController());

			if (ctx.isStopped()) {
				return;
			}

			if (!ctx.isDontGoDeeper()) {
				// visit the value of the field
				var field = state.getField();
				var instance = state.value;

				boolean shouldVisit = true;
				// don't visit if primitive
				shouldVisit &= !field.getType().isPrimitive();
				// don't visit static fields
				shouldVisit &= !Modifier.isStatic(field.getModifiers());

				if (shouldVisit && !field.canAccess(instance)) {
					// don't visit if we can't access the field value
					shouldVisit &= field.trySetAccessible();
				}

				if (shouldVisit) {
					try {
						Object fieldValue = field.get(instance);
						ctx.pushOnStack(fieldValue, State.OBJECT);
						return;
					} catch (IllegalArgumentException | IllegalAccessException e) {
						// NOTE should we log this?
					}
				}
				// we could not visit the field
			} else {
				// we are done with this field
			}
		} else {
			// coming back from a field visit.
			if (ctx.isStopped()) {
				return;
			}
		}

		// try to go to the next field
		// set state to continue
		ctx.resetControl();

		if (state.tryGoNextField()) {
			return;
		}

		ctx.goBack();
		GraphWalkerClassState classState = (GraphWalkerClassState) ctx.current();
		assert classState != null : "Illegal state";
		classState.fieldsDone();
	}
}
