package de.serra.graph_walker;

import de.serra.graph_walker.GraphWalkerContext.VisitController;

import org.checkerframework.checker.nullness.qual.Nullable;

import java.lang.reflect.Field;
import java.util.List;
import java.util.Map;

/**
 * Visitor for the object graph.
 * <p>
 * The methods will be called as specific parts in the object graph are reached. The visits follow a depth-first
 * approach.
 * <p>
 * All methods accept a {@link VisitController}. Use it to control the traversal.
 *
 * @author Peter Lamby
 * @see VisitController
 */
public interface ObjectgraphVisitor {
	/**
	 * Called when a {@code byte[]} is encountered.
	 *
	 * @param value   the byte array.
	 * @param control the control.
	 */
	default void visitByteArray(byte[] value, VisitController control) {
	}

	/**
	 * Called when a {@code short[]} is encountered.
	 *
	 * @param value   the short array.
	 * @param control the control.
	 */
	default void visitShortArray(short[] value, VisitController control) {
	}

	/**
	 * Called when an {@code int[]} is encountered.
	 *
	 * @param value   the int array.
	 * @param control the control.
	 */
	default void visitIntArray(int[] value, VisitController control) {
	}

	/**
	 * Called when a {@code long[]} is encountered.
	 *
	 * @param value   the long array.
	 * @param control the control.
	 */
	default void visitLongArray(long[] value, VisitController control) {
	}

	/**
	 * Called when a {@code float[]} is encountered.
	 *
	 * @param value   the float array.
	 * @param control the control.
	 */
	default void visitFloatArray(float[] value, VisitController control) {
	}

	/**
	 * Called when a {@code double[]} is encountered.
	 *
	 * @param value   the double array.
	 * @param control the control.
	 */
	default void visitDoubleArray(double[] value, VisitController control) {
	}

	/**
	 * Called when a {@code boolean[]} is encountered.
	 *
	 * @param value   the boolean array.
	 * @param control the control.
	 */
	default void visitBooleanArray(boolean[] value, VisitController control) {
	}

	/**
	 * Called when a {@code char[]} is encountered.
	 *
	 * @param value   the char array.
	 * @param control the control.
	 */
	default void visitCharArray(char[] value, VisitController control) {
	}

	/**
	 * Called when a {@code null} value is encountered.
	 *
	 * @param control the control.
	 */
	default void visitNull(VisitController control) {
	}

	/**
	 * Called when any value is encountered.
	 * <p>
	 * This means this method is called once before most of the specific visit methods here are called.
	 * <p>
	 * It is <em>not</em> called before the leave.. and following methods:
	 * <ul>
	 * <li>{@link #visitClassField(Object, Field, VisitController)}</li>
	 * <li>{@link #visitSuperClass(Object, Class, VisitController)}</li>
	 * </ul>
	 * <p>
	 * The visit will usually decend further into the objectgraph after this method. To avoid infinite traversing we will
	 * <em>not</em> continue if the already saw the object in the current path.
	 *
	 * @param value       the value.
	 * @param control     the control.
	 * @param alreadySeen {@code true} if we already saw this object. {@code false} otherwise.
	 */
	default void beforeObject(@Nullable Object value, VisitController control, boolean alreadySeen) {
	}

	/**
	 * Not yet implemented.
	 *
	 * @param value   the value.
	 * @param control the control.
	 */
	default void visitEnum(Enum<?> value, VisitController control) {
	}

	/**
	 * Called when visiting "normal" class.
	 * <p>
	 * Called when the {@code value} is not already handled bye
	 * <ul>
	 * <li>The primitive array methods</li>
	 * <li>{@link #visitArray(Object[], VisitController)}</li>
	 * <li>{@link #visitNull(VisitController)}</li>
	 * </ul>
	 *
	 * @param value   the value.
	 * @param control the control.
	 */
	default void visitClass(Object value, VisitController control) {
	}

	/**
	 * Called when visiting a field in a class.
	 * <p>
	 * Note that {@code value} is <em>not</em> the value of the field. It is the instance which contains {@code field}.
	 * <p>
	 * This is called directly after {@link #visitClass(Object, VisitController)}. Only the members that are declared in
	 * the specific class will be visited.
	 * <p>
	 * If there are parent classes {@link #visitSuperClass(Object, Class, VisitController)} will be called for each
	 * parent and this method will be called after each step in the class hierarchy.
	 * <p>
	 * The following fields (and their values) will be ignored and not visited:
	 * <ul>
	 * <li>primitives like {@code int} or {@code char}</li>
	 * <li>static fields</li>
	 * <li>fields that are not accessible and can't be made acessible</li>
	 * </ul>
	 *
	 * @param value   the instance to which {@code field} belongs.
	 * @param field   The field inside {@code value}.
	 * @param control the control.
	 * @see #visitClass(Object, VisitController)
	 * @see #visitSuperClass(Object, Class, VisitController)
	 */
	default void visitClassField(Object value, Field field, VisitController control) {
	}

	/**
	 * Called when visiting the parent of a class.
	 * <p>
	 * Called after all {@link #visitClassField(Object, Field, VisitController) fields} have been already visited.
	 *
	 * @param value     the value.
	 * @param superType the current type in the class hierarchy.
	 * @param control   the control.
	 */
	default void visitSuperClass(Object value, Class<?> superType, VisitController control) {
	}

	/**
	 * Called when visiting of a class has finished.
	 * <p>
	 * This means {@link #visitClass(Object, VisitController)} {@link #visitClassField(Object, Field, VisitController)}
	 * and {@link #visitSuperClass(Object, Class, VisitController)} have already handled all values.
	 *
	 * @param value   the value.
	 * @param control the control.
	 */
	default void leaveClass(Object value, VisitController control) {
	}

	/**
	 * Called when an non primitive array is encountered.
	 *
	 * @param value   the array.
	 * @param control the control.
	 */
	default void visitArray(@Nullable Object[] value, VisitController control) {
	}

	/**
	 * Called for every entry in an non primitive array.
	 *
	 * @param value   the value.
	 * @param idx     the index of {@code value} in {@code array}.
	 * @param array   the entire array.
	 * @param control the control.
	 */
	default void visitArrayMember(@Nullable Object value, int idx, @Nullable Object[] array, VisitController control) {
	}

	/**
	 * Called after all members in an array have been {@link #visitArrayMember(Object, int, Object[], VisitController)
	 * visited}.
	 *
	 * @param value   the array.
	 * @param control the control.
	 */
	default void leaveArray(@Nullable Object[] value, VisitController control) {
	}

	/**
	 * Not yet implemented.
	 *
	 * @param value   the value.
	 * @param control the control.
	 */

	default void visitNonListIterable(Iterable<?> value, VisitController control) {
	}

	/**
	 * Not yet implemented.
	 *
	 * @param value   the value.
	 * @param control the control.
	 */

	default void visitIterableMember(Object value, VisitController control) {
	}

	/**
	 * Not yet implemented.
	 *
	 * @param value   the value.
	 * @param control the control.
	 */

	default void leaveIterable(Object value, VisitController control) {
	}

	/**
	 * Not yet implemented.
	 *
	 * @param value   the value.
	 * @param control the control.
	 */

	default void visitList(List<?> value, VisitController control) {
	}

	/**
	 * Not yet implemented.
	 *
	 * @param value   the value.
	 * @param idx     todo
	 * @param list    todo
	 * @param control the control.
	 */

	default void visitListMember(Object value, int idx, List<?> list, VisitController control) {
	}

	/**
	 * Not yet implemented.
	 *
	 * @param value   the value.
	 * @param control the control.
	 */

	default void leaveList(List<?> value, VisitController control) {
	}

	/**
	 * Not yet implemented.
	 *
	 * @param value   the value.
	 * @param control the control.
	 */

	default void visitMap(Map<?, ?> value, VisitController control) {
	}

	/**
	 * Not yet implemented.
	 *
	 * @param key     todo
	 * @param value   the value.
	 * @param map     todo
	 * @param control the control.
	 */

	default void visitMapEntry(Object key, Object value, Map<?, ?> map, VisitController control) {
	}

	/**
	 * Not yet implemented.
	 *
	 * @param value   the value.
	 * @param control the control.
	 */

	default void leaveMap(Map<?, ?> value, VisitController control) {
	}
}
