package de.serra.graph_walker;

import de.serra.graph_walker.GraphWalkerState.State;

import org.checkerframework.checker.initialization.qual.NotOnlyInitialized;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.checkerframework.dataflow.qual.Pure;

import java.util.Arrays;

class GraphWalkerContext {
	public Control control = Control.CONTINUE;

	final ObjectgraphVisitor visitor;

	private @Nullable GraphWalkerState<?>[] stack = new GraphWalkerState[16];
	private int stackSize;
	private final @NotOnlyInitialized VisitController controller;

	GraphWalkerContext(Object start, final ObjectgraphVisitor visitor) {
		this.visitor = visitor;
		stack[0] = new GraphWalkerState<Object>(State.OBJECT, start);
		stackSize++;
		controller = new VisitController(this);
	}

	VisitController getController() {
		return controller;
	}

	void goBack() {
		stackSize--;
	}

	boolean alreadySeen(Object o) {
		for (int i = stackSize - 2; i >= 0; i--) {
			assert stack[i] != null;

			if (stack[i].state == State.OBJECT && stack[i].value == o) {
				return true;
			}
		}
		return false;
	}

	@Pure
	@Nullable
	GraphWalkerState<?> current() {
		return relativeFromCurrent(0);
	}

	@Pure
	@Nullable
	GraphWalkerState<?> relativeFromCurrent(int offset) {
		assert stackSize > 0 + offset : "Illegal State. No more element in stack";
		return stack[stackSize - (offset + 1)];
	}

	void replaceCurrent(final GraphWalkerState<?> newState) {
		assert stackSize > 0 : "Cant replace current when stack is empty";
		stack[stackSize - 1] = newState;
	}

	void pushOnStack(final @Nullable Object o, final State state) {
		pushOnStack(new GraphWalkerState<@Nullable Object>(state, o));
	}

	void pushOnStack(final GraphWalkerState<?> state) {
		if (stackSize == stack.length) {
			grow(stackSize + 1);
		}
		stack[stackSize] = state;
		stackSize++;
	}

	/**
	 * Increases the capacity to ensure that it can hold at least the number of elements specified by the minimum
	 * capacity argument.
	 *
	 * @param minCapacity the desired minimum capacity
	 */
	private void grow(int minCapacity) {
		int oldCapacity = stack.length;
		int newCapacity = Math.max(
				// grow by half of current size.
				oldCapacity + oldCapacity >> 1, minCapacity);
		// realloc
		stack = Arrays.copyOf(stack, newCapacity);
	}

	void resetControl() {
		assert control != Control.STOP : "Should never try to continue after stop.";

		this.control = Control.CONTINUE;
	}

	@Pure
	boolean isDone() {
		return stackSize == 0 || control == Control.STOP;
	}

	/**
	 * Is the traversal stopped?
	 *
	 * @return {@code true} if the traversal is stopped.
	 */
	@Pure
	boolean isStopped() {
		return control == Control.STOP;
	}

	/**
	 * Should the children not be traversed.
	 *
	 * @return {@code true} if the children should not be traversed.
	 */
	@Pure
	boolean isDontGoDeeper() {
		return control == Control.STOP || control == Control.CONTINUE_BUT_DONT_GO_DEEPER;
	}

	/**
	 * Should the parent classes not be traversed.
	 *
	 * @return {@code true} if the parent classes should not be traversed.
	 */
	@Pure
	boolean isDontGoSuper() {
		return control == Control.STOP || control == Control.CONTINUE_BUT_DONT_GO_SUPER;
	}

	void leave() {
		goBack();
		if (!isDone()) {
			@Nullable
			GraphWalkerState<?> state = current();
			if (state != null) {
				state.leave();
			}

		}
	}

	enum Control {
		STOP, CONTINUE, CONTINUE_BUT_DONT_GO_DEEPER, CONTINUE_BUT_DONT_GO_SUPER;
	}
}
