package de.serra.graph_walker;

import org.checkerframework.dataflow.qual.Pure;

/**
 * Represents the method specific state at this point in the call stack.
 *
 * Supports Tail-Recursion by just modifying the current stack frame.
 *
 * @param <V> The type of the "main" value associated with this stack frame.
 */
public class GraphWalkerStackFrame<V> {
	private final State state;
	private final V value;
	private Direction direction = Direction.ENTERING;

	/**
	 * Constructs.
	 *
	 * @param state The State to use for dispatching to the correct method.
	 * @param value The "main" value associated with this stack frame.
	 */
	public GraphWalkerStackFrame(final State state, final V value) {
		this.value = value;
		this.state = state;
	}

	/**
	 * The state decides which method to use for dispatching to the correct method.
	 *
	 * @return The state.
	 */
	@Pure
	public State state() {
		return state;
	}

	/**
	 * The "main" value associated with this stack frame.
	 *
	 * @return The "main" value.
	 */
	@Pure
	public V value() {
		return value;
	}

	/**
	 * The types of methods available for dispatching.
	 */
	public enum State {
		/**
		 * Call beforeObject next.
		 */
		BEFORE_OBJECT,
		/**
		 * Visit an array next.
		 */
		ARRAY,
		/**
		 * Visit the member of an array next.
		 */
		ARRAY_MEMBER,
		/**
		 * Visit a "normal" object next.
		 */
		CLASS,
		/**
		 * Visit the field of a "normal" object next.
		 */
		CLASS_FIELD;
	}

	/**
	 * How we arrived at the {@link GraphWalkerStackFrame stack frame}.
	 */
	enum Direction {
		/**
		 * The stackframe was just created and is now used to call a method.
		 */
		ENTERING,
		/**
		 * The stack frame was called again after returning from somewhere.
		 */
		LEAVING;
	}

	/**
	 * Marks the stackframe as being returned to.
	 *
	 * @see #isLeaving()
	 */
	public void leave() {
		assert direction == Direction.ENTERING : "Already leaving";

		direction = Direction.LEAVING;
	}

	/**
	 * Is this stack frame is being returned to.
	 *
	 * @return {@code true} when this stack frame is being returned to. {@code false} otherwise.
	 */
	@Pure
	public boolean isLeaving() {
		return direction == Direction.LEAVING;
	}

	/**
	 * Sets the direction to entering again.
	 */
	protected void resetDirection() {
		direction = Direction.ENTERING;
	}
}
