package de.serra.so_dirty.sn;

import de.serra.so_dirty.difference.DifferenceNode;
import de.serra.so_dirty.difference.DifferenceNodes;
import de.serra.so_dirty.difference.DifferenceType;
import de.serra.so_dirty.difference.LeafDifferenceNode;

import org.checkerframework.checker.nullness.qual.Nullable;

import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import static de.serra.so_dirty.difference.DifferenceType.EQUALITY;
import static de.serra.so_dirty.difference.DifferenceType.REFERENCE;
import static de.serra.so_dirty.difference.DifferenceType.TYPE_CHANGE;

/**
 * Snapshot of an class and its fields.
 *
 * @author Peter Lamby
 */
public class ClassSnapshotNode implements SnapshotNode {
	private final Object value;
	private final HashMap<String, @Nullable SnapshotNode> fieldsToSnapshots;

	/**
	 * Constructs
	 *
	 * @param value             The instance of the class.
	 * @param fieldsToSnapshots a {@link Map} of field names to the corresponding {@link SnapshotNode}.
	 */
	public ClassSnapshotNode(final Object value, final Map<String, @Nullable SnapshotNode> fieldsToSnapshots) {
		this.value = value;
		this.fieldsToSnapshots = new HashMap<>(fieldsToSnapshots);
	}

	@Override
	public Object value() {
		return value;
	}

	@Override
	public DifferenceNode diff(final @Nullable SnapshotNode other, final String path) {
		final var diffTypes = EnumSet.noneOf(DifferenceType.class);
		if (other == null || !Objects.equals(value, other.value())) {
			diffTypes.add(DifferenceType.EQUALITY);
		}
		if (other == null || value != other.value()) {
			diffTypes.add(DifferenceType.REFERENCE);
		}
		if (other == null || !Objects.equals(value.getClass(), other.value().getClass())) {
			diffTypes.add(TYPE_CHANGE);
		}

		if (!(other instanceof ClassSnapshotNode)) {
			// Other is not a class anymore
			return new LeafDifferenceNode(path, diffTypes.toArray(DifferenceType[]::new));
		}

		final ClassSnapshotNode them = (ClassSnapshotNode) other;
		final var differences = new DifferenceNodes(path, '.', diffTypes.toArray(DifferenceType[]::new));

		// first we check every field that we know of.
		for (final Map.Entry<String, @Nullable SnapshotNode> entry : fieldsToSnapshots.entrySet()) {
			final var snapshotNode = entry.getValue();
			final var otherSnapshotNode = them.fieldsToSnapshots.get(entry.getKey());

			if (snapshotNode == null && otherSnapshotNode == null) {
				continue;
			}

			final DifferenceNode difference = SnapshotNode.doDiff(snapshotNode, otherSnapshotNode, entry.getKey());
			if (difference.isDifferentRecursive((DifferenceType[]) null)) {
				// only record actual differences
				differences.add(difference);
			}
		}

		// We alread checked every field we know of obove. Now we only need to check which new fields wie don't know of.
		for (final String key : them.fieldsToSnapshots.keySet()) {
			if (!fieldsToSnapshots.containsKey(key)) {
				differences.add(new LeafDifferenceNode(key, TYPE_CHANGE, REFERENCE, EQUALITY));
			}
		}

		return differences;
	}
}
