package de.serra.so_dirty.sn;

import de.serra.so_dirty.difference.DifferenceNode;
import de.serra.so_dirty.difference.DifferenceType;
import de.serra.so_dirty.difference.LeafDifferenceNode;
import de.serra.so_dirty.difference.MapDifferenceNode;

import org.checkerframework.checker.nullness.qual.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Objects;

import static de.serra.so_dirty.difference.DifferenceType.EQUALITY;
import static de.serra.so_dirty.difference.DifferenceType.KEY_ADDED;
import static de.serra.so_dirty.difference.DifferenceType.KEY_REMOVED;
import static de.serra.so_dirty.difference.DifferenceType.REFERENCE;
import static de.serra.so_dirty.difference.DifferenceType.TYPE_CHANGE;

/**
 * Snapshot of a map and its entries.
 *
 * @author Peter Lamby
 */
public class MapSnapshotNode implements SnapshotNode {
	private final Object value;
	private final ArrayList<MapMemberSnapshotNode> entries;

	/**
	 * Constructs.
	 *
	 * @param value   The map instance.
	 * @param entries The entries of the map.
	 */
	public MapSnapshotNode(final Object value, final Collection<MapMemberSnapshotNode> entries) {
		this.value = value;
		this.entries = new ArrayList<>(entries);
	}

	@Override
	public Object value() {
		return value;
	}

	@Override
	public DifferenceNode diff(final @Nullable SnapshotNode other, final String path) {
		final var diffTypes = EnumSet.noneOf(DifferenceType.class);
		if (other == null || !Objects.equals(value, other.value())) {
			diffTypes.add(EQUALITY);
		}
		if (other == null || value != other.value()) {
			diffTypes.add(REFERENCE);
		}
		if (other == null || !Objects.equals(value.getClass(), other.value().getClass())) {
			diffTypes.add(TYPE_CHANGE);
		}

		if (!(other instanceof MapSnapshotNode)) {
			// Other is not a map anymore
			return new LeafDifferenceNode(path, diffTypes.toArray(DifferenceType[]::new));
		}

		final MapSnapshotNode them = (MapSnapshotNode) other;

		// NOTE maybe dont record entries at all when we can't snapshot the key?
		int foundEntries = 0;
		int removedKeys = 0;
		int addedKeys = 0;
		HashMap<@Nullable Object, DifferenceNode> differences = new HashMap<>();

		for (int i = 0; i < entries.size(); i++) {
			final var oldEntry = entries.get(i);
			final @Nullable SnapshotNode oldKey = oldEntry.keySnapshot;
			final @Nullable MapMemberSnapshotNode newEntry = findByKey(oldKey == null ? null : oldKey.value(),
					them.entries);
			if (newEntry == null) {
				removedKeys--;
				diffTypes.add(KEY_REMOVED);
			} else {
				foundEntries++;
				final var mapIdx = "[" + i + "]";
				DifferenceNode valueDiff = SnapshotNode.doDiff(oldEntry.valueSnapshot, newEntry.valueSnapshot, mapIdx);
				if (valueDiff.isDifferent((DifferenceType[]) null)) {
					// only record actual differences
					differences.put(oldKey == null ? null : oldKey.value(), valueDiff);
				}
			}
		}

		// check for entries not known to us
		if (foundEntries < them.entries.size()) {
			diffTypes.add(KEY_ADDED);
			for (var newEntry : them.entries) {
				var newKey = newEntry.keySnapshot;
				if (findByKey(newKey == null ? null : newKey.value(), entries) == null) {
					addedKeys++;
				}
			}
			assert addedKeys > 0 : "we didn't find all entries but also could not determin the new ones";
		}

		return new MapDifferenceNode(path, diffTypes, removedKeys, addedKeys, differences);
	}

	private @Nullable MapMemberSnapshotNode findByKey(final @Nullable Object needle,
			final ArrayList<MapMemberSnapshotNode> hayball)
	{
		for (final var curr : hayball) {
			if (curr.keySnapshot == null) {
				if (needle == null) {
					return curr;
				}
				continue;
			}
			if (Objects.equals(needle, curr.keySnapshot.value())) {
				return curr;
			}
		}
		return null;
	}

	/**
	 * Snapshot of a single entry of a map.
	 *
	 * @author Peter Lamby
	 */
	public static class MapMemberSnapshotNode {
		/**
		 * The key snapshot. Should be {@code null} if the key itself is {@code null}.
		 */
		public final @Nullable SnapshotNode keySnapshot;
		/**
		 * The value snapshot. Should be {@code null} if the value itself is {@code null}
		 */
		public final @Nullable SnapshotNode valueSnapshot;

		/**
		 * Constructs
		 *
		 * @param keySnapshot   The key snapshot. Should be {@code null} if the key itself is {@code null}.
		 * @param valueSnapshot The value snapshot. Should be {@code null} if the value itself is {@code null}.
		 */
		public MapMemberSnapshotNode(final @Nullable SnapshotNode keySnapshot,
				final @Nullable SnapshotNode valueSnapshot)
		{
			this.keySnapshot = keySnapshot;
			this.valueSnapshot = valueSnapshot;
		}
	}
}
