package de.serra.so_dirty.difference;

import de.serra.so_dirty.difference.visit.CollectDifferencesVisitor;
import de.serra.so_dirty.difference.visit.DifferenceVisitor;
import de.serra.so_dirty.difference.visit.GetChildVisitor;
import de.serra.so_dirty.difference.visit.IsDifferentVisitor;
import de.serra.so_dirty.util.PathUtil;

import org.checkerframework.checker.nullness.qual.Nullable;

import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.Set;

/**
 * The result of a diff operation. Represents a single node in the object tree.
 *
 * @author Peter Lamby
 */
public interface DifferenceNode extends Iterable<DifferenceNode> {
	/**
	 * Visits the current node.
	 * <p>
	 * The children will be visited after {@code this} node has been visited.
	 * <p>
	 * The visiting of children can be prevented by calling {@link DifferenceVisit#dontGoDeeper()} usually from the
	 * {@code visitor}.
	 *
	 * @param <T>     The return type of the visit.
	 * @param visitor The visitor.
	 * @param visit   The visit.
	 * @return The {@link DifferenceVisit#value()}.
	 */
	<T> T visit(DifferenceVisitor<T> visitor, DifferenceVisit<T> visit);

	/**
	 * The relative path inside the object tree.
	 *
	 * @return The path.
	 */
	String getName();

	/**
	 * Describes in what way the current node changed. Returns an empty set if there are no changes.
	 *
	 * @return The types of change if any.
	 */
	Set<DifferenceType> getTypes();

	/**
	 * All the {@link DifferenceType} in the hierarchy.
	 *
	 * @return all the {@link DifferenceType} in the hierarchy.
	 */
	default Set<DifferenceType> getTypesRecursive() {
		var sharedState = EnumSet.noneOf(DifferenceType.class);
		return visit(new CollectDifferencesVisitor(sharedState), new DifferenceVisit<>(sharedState));
	}

	/**
	 * Gets a differenceNode specified by an expression.
	 * <p>
	 * There is no specification for the expression language right now. Please take a look at the PathUtilTest to see how
	 * to use it.
	 *
	 * @param path The path expression.
	 * @return The difference node at the specified path.
	 */
	default DifferenceNode get(final String path) {
		if (path.isBlank()) {
			return new EmptyDifferenceNode(path);
		}

		final var absolutPath = path.startsWith(".") ? path : "." + path;
		return visit(new GetChildVisitor(absolutPath),
				new DifferenceVisit<>(new EmptyDifferenceNode(PathUtil.getLastPart(absolutPath))));
	}

	/**
	 * Gets a direct child node in the object tree.
	 * <p>
	 * Will return {@code null} if the child does not exist.
	 *
	 * @param name The name of the child node.
	 * @return The result of the desired node. May return {@code null}.
	 */
	@Nullable
	DifferenceNode getChildNullable(String name);

	/**
	 * Gets a direct child node in the object tree.
	 * <p>
	 * Will always successfully return even if the node does not exist. In that case an empty result is returned.
	 *
	 * @param childName The name of the child node.
	 * @return The result of the desired node. Will return a result with no changes if the node does not exist.
	 */
	default DifferenceNode getChild(final String childName) {
		final var child = getChildNullable(childName);
		return child != null ? child : new EmptyDifferenceNode(childName);
	}

	/**
	 * Did the node change in one of the ways described by types.
	 * <p>
	 * This method will only report a difference if it's one of the {@code types}. If {@code types} is {@code null} this
	 * methods checks for any difference. If {@code types.length == 0} this method will always return false.
	 *
	 * @param types The types of difference to check for.
	 * @return {@code true} if there is a difference described by {@code types} {@code false} otherwise.
	 */
	default boolean isDifferent(final DifferenceType @Nullable... types) {
		if (types == null) {
			return !getTypes().isEmpty();
		}
		if (types.length == 0) {
			return false;
		}
		return !Collections.disjoint(getTypes(), Arrays.asList(types));
	}

	/**
	 * Convenience method for
	 * {@snippet lang = "java" :
	 * // @link substring="getChild" target="#getChild(String)" @link substring="isDifferent" target="#isDifferent(DifferenceType...)" :
	 * differenceNode.getChild("myPath").isDifferent(types...);
	 * }
	 *
	 * @param path  The relative path of the desired node to be checked.
	 * @param types The types of difference to check for.
	 * @return {@code true} if there is a difference described by {@code types} {@code false} otherwise.
	 */
	default boolean isDifferent(final String path, final DifferenceType @Nullable... types) {
		return getChild(path).isDifferent(types);
	}

	/**
	 * Did the current node or one of its children change?
	 * <p>
	 * This method will only report a difference if it's one of the {@code types}. If {@code types} is {@code null} this
	 * methods checks for any difference. If {@code types.length == 0} this method will always return false.
	 *
	 * @param types The types of difference to check for.
	 * @return {@code true} if the current node or its children changed. {@code false} otherwise.
	 */
	default boolean isDifferentRecursive(final DifferenceType @Nullable... types) {
		if (types != null && types.length == 0) {
			return false;
		}
		return visit(new IsDifferentVisitor(types), new DifferenceVisit<>(Boolean.FALSE));
	}

	/**
	 * A Difference that is always empty.
	 *
	 * @author Peter Lamby
	 */
	class EmptyDifferenceNode implements DifferenceNode {
		private final String name;

		/**
		 * Constructs.
		 *
		 * @param name The name of this node.
		 */
		public EmptyDifferenceNode(final String name) {
			this.name = name;
		}

		@Override
		public <T> T visit(DifferenceVisitor<T> visitor, DifferenceVisit<T> visit) {
			visitor.visitDifferenceNode(this, visit.appendPathPrefix(name), visit);
			return visit.value();
		}

		@Override
		public Iterator<DifferenceNode> iterator() {
			return Collections.emptyIterator();
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public Set<DifferenceType> getTypes() {
			return Collections.emptySet();
		}

		@Override
		public @Nullable DifferenceNode getChildNullable(final String name) {
			return null;
		}
	}
}
