package de.serra.so_dirty.difference;

import de.serra.so_dirty.difference.visit.DifferenceVisitor;
import de.serra.so_dirty.util.UnmodifiableIterator;

import org.checkerframework.checker.nullness.qual.Nullable;

import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

/**
 * Multiple differences. Like an array or a class with fields.
 *
 * @author Peter Lamby
 */
public class DifferenceNodes implements DifferenceNode {
	private final String name;
	private final @Nullable Set<DifferenceType> types;
	private final HashMap<String, DifferenceNode> pathsToDifferences = new HashMap<>();
	private final @Nullable Character pathSeparator;

	/**
	 * Constructs.
	 *
	 * @param name          The relative path of this difference.
	 * @param pathSeparator The separator to include when visiting.
	 * @param types         The types of difference for this node. Can be {@code null}.
	 */
	public DifferenceNodes(final String name, final @Nullable Character pathSeparator,
			final DifferenceType @Nullable... types)
	{
		this.name = name;
		this.pathSeparator = pathSeparator;
		if (types == null || types.length == 0) {
			this.types = null;
		} else {
			this.types = EnumSet.noneOf(DifferenceType.class);
			this.types.addAll(Arrays.asList(types));
		}
	}

	/**
	 * Constructs.
	 *
	 * @param name  The relative path of this difference.
	 * @param types The types of difference for this node. Can be {@code null}.
	 */
	public DifferenceNodes(final String name, final DifferenceType @Nullable... types) {
		this(name, null, types);
	}

	@Override
	public <T> T visit(DifferenceVisitor<T> visitor, DifferenceVisit<T> visit) {
		visitor.visitDifferenceNode(this, visit.appendPathPrefix(name), visit);
		if (visit.isDontGoDeeper()) {
			return visit.value();
		}

		final CharSequence currentPath;
		if (pathSeparator != null && visit.pathPrefix().charAt(visit.pathPrefix().length() - 1) != pathSeparator) {
			currentPath = new StringBuilder(visit.pathPrefix()).append(pathSeparator.charValue());
		} else {
			currentPath = visit.pathPrefix();
		}
		for (var child : this) {
			// new visit to reset the action/state.
			var childTraversal = new DifferenceVisit<>(visit.value(), currentPath);
			child.visit(visitor, childTraversal);
			if (childTraversal.isStopped()) {
				visit.stop(childTraversal.value());
				break;
			}
		}
		return visit.value();
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public Set<DifferenceType> getTypes() {
		return types == null ? Collections.emptySet() : Collections.unmodifiableSet(types);
	}

	@Override
	public @Nullable DifferenceNode getChildNullable(String name) {
		return pathsToDifferences.get(name);
	}

	/**
	 * Add a child difference.
	 *
	 * @param difference The child.
	 */
	public void add(final DifferenceNode difference) {
		if (pathsToDifferences.put(difference.getName(), difference) != null) {
			throw new IllegalStateException("Trying to add a difference for the same path twice");
		}
	}

	@Override
	public Iterator<DifferenceNode> iterator() {
		return new UnmodifiableIterator<>(pathsToDifferences.values().iterator());
	}
}
