package de.serra.so_dirty.difference;

import de.serra.so_dirty.difference.visit.DifferenceVisitor;
import de.serra.so_dirty.util.UnmodifiableIterator;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import org.checkerframework.checker.nullness.qual.Nullable;

import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * Differences of a {@link Map}.
 *
 * @author Peter Lamby
 */
public class MapDifferenceNode implements DifferenceNode {
	private final String name;
	private final Set<DifferenceType> types;
	private final int removedKeys;
	private final int addedKeys;
	private final Map<@Nullable Object, DifferenceNode> differentValues;

	/**
	 * Constructs.
	 *
	 * @param name            The relative path of this difference.
	 * @param types           The types of difference for this node. Can be {@code null}.
	 * @param removedKeys     How many keys were removed.
	 * @param addedKeys       How many keys were added.
	 * @param differentValues The map entries where the value has changed.
	 */
	@SuppressFBWarnings(
			value = "FCBL_FIELD_COULD_BE_LOCAL",
			justification = "We keep removed/added keys for now to be able to expose them as difference later on")
	public MapDifferenceNode(final String name, @Nullable Set<DifferenceType> types, int removedKeys, int addedKeys,
			Map<@Nullable Object, DifferenceNode> differentValues)
	{
		this.name = name;
		this.types = types == null ? Collections.emptySet() : EnumSet.copyOf(types);
		this.removedKeys = removedKeys;
		this.addedKeys = addedKeys;
		this.differentValues = new HashMap<>(differentValues);
	}

	@Override
	public Iterator<DifferenceNode> iterator() {
		return new UnmodifiableIterator<>(differentValues.values().iterator());
	}

	@Override
	public <T> T visit(DifferenceVisitor<T> visitor, DifferenceVisit<T> visit) {
		visitor.visitMapDifferenceNode(this, visit.appendPathPrefix(name), visit);
		if (visit.isDontGoDeeper()) {
			return visit.value();
		}

		for (var child : this) {
			// new visit to reset the action/state.
			var childTraversal = new DifferenceVisit<>(visit.value(), visit.pathPrefix());
			child.visit(visitor, childTraversal);
			if (childTraversal.isStopped()) {
				visit.stop(childTraversal.value());
				break;
			}
		}
		return visit.value();
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public Set<DifferenceType> getTypes() {
		return Collections.unmodifiableSet(types);
	}

	@Override
	public @Nullable DifferenceNode getChildNullable(String name) {
		// NOTE we can't get map children by string.
		return null;
	}
}
