package de.serra.so_dirty.difference;

/**
 * Allows visitors to control the traversal.
 *
 * @author Peter Lamby
 * @param <T> the type of object the visitor is expected to return. If none use {@link Void}.
 */
public class DifferenceVisit<T> {
	private final StringBuilder pathPrefixBuffer;
	private Action action = Action.CONTINUE;
	private T value;

	/**
	 * Constructs.
	 * <p>
	 * Allows to specify a default {@code value} that is returned by {@link #value()} if no more specifc is set by
	 * {@link #stop(Object)}.
	 *
	 * @param value The default value.
	 */
	public DifferenceVisit(final T value) {
		this.value = value;
		this.pathPrefixBuffer = new StringBuilder();
	}

	/**
	 * Constructs.
	 * <p>
	 * Allows to specify a default {@code value} that is returned by {@link #value()} if no more specifc is set by
	 * {@link #stop(Object)}.
	 *
	 * @param value The default value.
	 * @param path  The current pathPrefix of the visit.
	 */
	DifferenceVisit(T value, final CharSequence path) {
		this.value = value;
		this.pathPrefixBuffer = new StringBuilder(path);
	}

	/**
	 * Stops the traversal.
	 */
	public void stop() {
		this.action = Action.STOP;
	}

	/**
	 * Stops the traversal with {@code value}.
	 *
	 * @param value The return value of the traversal.
	 */
	public void stop(T value) {
		this.action = Action.STOP;
		this.value = value;
	}

	/**
	 * Avoids the traversal of children.
	 */
	public void dontGoDeeper() {
		this.action = Action.CONTINUE_BUT_DONT_GO_DEEPER;
	}

	/**
	 * The value that should be returned by the traversal.
	 *
	 * @return the value of the traversal.
	 */
	public T value() {
		return value;
	}

	/**
	 * Is the traversal stopped?
	 *
	 * @return {@code true} if the traversal is stopped.
	 */
	public boolean isStopped() {
		return action == Action.STOP;
	}

	/**
	 * Should the children not be traversed.
	 *
	 * @return {@code true} if the children should not be traversed.
	 */
	public boolean isDontGoDeeper() {
		return action == Action.STOP || action == Action.CONTINUE_BUT_DONT_GO_DEEPER;
	}

	String appendPathPrefix(String path) {
		pathPrefixBuffer.append(path);
		return pathPrefixBuffer.toString();
	}

	CharSequence pathPrefix() {
		return pathPrefixBuffer;
	}

	enum Action {
		STOP, CONTINUE, CONTINUE_BUT_DONT_GO_DEEPER;
	}
}
