package de.serra.so_dirty.sn;

import org.checkerframework.checker.nullness.qual.Nullable;

import java.util.ArrayList;
import java.util.Arrays;

/**
 * {@link SnapshotNodeFactory} that delegates to an inner list of factories.
 *
 * @author Peter Lamby
 */
public class CompoundSnapshotNodeFactory implements SnapshotNodeFactory {
	private final ArrayList<SnapshotNodeFactory> inner = new ArrayList<>();

	/**
	 * Constructs.
	 *
	 * @param inner The actual factories.
	 */
	public CompoundSnapshotNodeFactory(final SnapshotNodeFactory @Nullable... inner) {
		if (inner != null && inner.length == 0) {
			this.inner.addAll(Arrays.asList(inner));
		}
	}

	/**
	 * Add a factory to the list of factories.
	 *
	 * @param factory The factory to add.
	 */
	public void add(final SnapshotNodeFactory factory) {
		this.inner.add(factory);
	}

	@Override
	public boolean supports(final Class<?> type) {
		for (final var factory : inner) {
			if (factory.supports(type)) {
				return true;
			}
		}
		return false;
	}

	@Override
	public @Nullable SnapshotNode toSnapshotNode(final Object value) {
		for (final var factory : inner) {
			if (factory.supports(value.getClass())) {
				return factory.toSnapshotNode(value);
			}
		}
		throw new IllegalArgumentException(
				value.getClass() + " is not supported. (Make sure to call #supports(Class) first.)");
	}
}
