package de.serra.so_dirty.sn;

import de.serra.so_dirty.difference.DifferenceNode;
import de.serra.so_dirty.difference.DifferenceType;
import de.serra.so_dirty.difference.LeafDifferenceNode;

import org.checkerframework.checker.nullness.qual.Nullable;

/**
 * Snapshot of an object.
 *
 * @author Peter Lamby
 */
public interface SnapshotNode {
	/**
	 * The value that was used to create the snapshot.
	 * <p>
	 * Please not that the interior of the object may have been modified since the snapshot was taken.
	 *
	 * @return The value that was used to create the snapshot.
	 */
	Object value();

	/**
	 * Compare this node to the {@code other} one.
	 * <p>
	 * {@code path} is used to provide context to the returned {@link DifferenceNode}.
	 *
	 * @param other The other node.
	 * @param path  The relative path of {@code this} and {@code other}.
	 * @return a {@link DifferenceNode} describing the differences. Never {@code null}.
	 */
	DifferenceNode diff(@Nullable SnapshotNode other, String path);

	/**
	 * Helper to compare two {@link SnapshotNode}s that are {@code null} safe.
	 * <p>
	 * {@code path} is used to provide context to the returned {@link DifferenceNode}.
	 *
	 * @param a    The node to compare.
	 * @param b    The node to compare.
	 * @param path The relative path of {@code a} and {@code b}.
	 * @return a {@link DifferenceNode} describing the differences.
	 */
	static DifferenceNode doDiff(final @Nullable SnapshotNode a, final @Nullable SnapshotNode b, final String path) {
		if (a != null) {
			return a.diff(b, path);
		}
		if (b != null) {
			return b.diff(null, path);
		}

		return new LeafDifferenceNode(path, (DifferenceType[]) null);
	}
}
