package de.serra.so_dirty;

import de.serra.so_dirty.difference.DifferenceNode;
import de.serra.so_dirty.sn.DefaultSnapshotNodeFactory;
import de.serra.so_dirty.sn.SnapshotNode;
import de.serra.so_dirty.sn.SnapshotNodeFactory;

import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;

/**
 * Creates a snapshot of an java object that can then be compared to another snappshot to check what has changed.
 *
 * {@snippet lang = "java" :
 * var myObject = new MyObject();
 * // @link substring="of" target="#of(Object)" @link substring="Snapshot" target="Snapshot" :
 * var previous = Snapshot.of(myObject);
 * // modify myObject here...
 * var differenceResult = previous.doDiff(myObject); // @link substring = "doDiff" target = "#doDiff(Object)"
 * // depending on MyObject and the modifications you can check for differences
 * // @link substring="isDifferent" target="DifferenceNode#isDifferent(String)" :
 * assertTrue(differenceResult.isDifferent("myField"), "Check if myField is different");
 * }
 *
 * @author Peter Lamby
 * @param <T> The type that is being checked for differences.
 */
public class Snapshot<T> {
	private final @Nullable SnapshotNode originalSnapshot;
	private final SnapshotNodeFactory factory;

	private Snapshot(final @Nullable SnapshotNode originalFieldValues, final SnapshotNodeFactory factory) {
		this.originalSnapshot = originalFieldValues;
		this.factory = factory;
	}

	/**
	 * Diff {@code this} against {@code other}.
	 * <p>
	 * Convenience method for
	 * {@snippet lang = "java" :
	 * // @link substring="doDiff" target="#doDiff(Snapshot)" @link substring="Snapshot" target="Snapshot" @link substring="of" target="Snapshot#of(Object, SnapshotNodeFactory)" :
	 * this.doDiff(Snapshot.of(other, this.factory));
	 * }
	 * Uses the same {@link SnapshotNodeFactory} that was used for {@code this}.
	 * <p>
	 * If you want to use a different {@link SnapshotNodeFactory} use {@link #doDiff(Snapshot)}.
	 *
	 * @param other The object to diff against.
	 * @return The differences.
	 */
	public DifferenceNode doDiff(final @NonNull T other) {
		final var newDiff = Snapshot.<T>of(other, factory);
		return doDiff(newDiff);
	}

	/**
	 * Diff {@code this} against {@code other}.
	 *
	 * @param other The snapshot to diff against.
	 * @return The differences.
	 */
	public DifferenceNode doDiff(final @NonNull Snapshot<T> other) {
		return SnapshotNode.doDiff(originalSnapshot, other.originalSnapshot, ".");
	}

	/**
	 * Create a snapshot of {@code val} using {@code factory}. If {@code val} may be {@code null} use
	 * {@link #of(Object, Class)} instead of this.
	 *
	 * @param <T>     The type of val.
	 * @param val     The value to snapshot.
	 * @param factory The factory that should be used to create the snapshot.
	 * @return The snapshot.
	 */
	public static <T> Snapshot<T> of(final @NonNull T val, final SnapshotNodeFactory factory) {
		return new Snapshot<T>(toSnapshotNode(val, val.getClass(), factory), factory);
	}

	/**
	 * Create a snapshot of {@code val}.
	 * <p>
	 * Convenience method for
	 * {@snippet lang = "java" :
	 * // @link substring="Snapshot" target="Snapshot" @link substring="of" target="#of(Object, SnapshotNodeFactory)" @link substring="DefaultSnapshotNodeFactory" target="DefaultSnapshotNodeFactory" @link substring="INSTANCE" target="DefaultSnapshotNodeFactory#INSTANCE" :
	 * Snapshot.of(val, DefaultSnapshotNodeFactory.INSTANCE);
	 * }
	 *
	 * @param <T> The type of val.
	 * @param val The value to snapshot.
	 * @return The snapshot.
	 */
	public static <T> Snapshot<T> of(final @NonNull T val) {
		return of(val, DefaultSnapshotNodeFactory.INSTANCE);
	}

	/**
	 * Create a snapshot of {@code val} using {@code factory}. If {@code val} may be {@code null} use this instead of
	 * {@link #of(Object)}.
	 *
	 * @param <T>     The type of val.
	 * @param val     The Instance to snapshot.
	 * @param type    The type to snapshot.
	 * @param factory The factory that should be used to create the snapshot.
	 * @return The snapshot.
	 */
	public static <T> Snapshot<T> of(final @Nullable T val, final Class<T> type, final SnapshotNodeFactory factory) {
		return new Snapshot<>(toSnapshotNode(val, type, factory), factory);
	}

	/**
	 * Create a snapshot of {@code val}. If {@code val} may be {@code null} use this instead of {@link #of(Object)}.
	 * <p>
	 * Convenience method for
	 * {@snippet lang = "java" :
	 * // @link substring="Snapshot" target="Snapshot" @link substring="of" target="#of(Object, Class, SnapshotNodeFactory)" @link substring="DefaultSnapshotNodeFactory" target="DefaultSnapshotNodeFactory" @link substring="INSTANCE" target="DefaultSnapshotNodeFactory#INSTANCE" :
	 * Snapshot.of(val, type, DefaultSnapshotNodeFactory.INSTANCE);
	 * }
	 *
	 * @param <T>  The type of val.
	 * @param val  The Instance to snapshot.
	 * @param type The type to snapshot.
	 * @return The snapshot.
	 */
	public static <T> Snapshot<T> of(final @Nullable T val, final Class<T> type) {
		return of(val, type, DefaultSnapshotNodeFactory.INSTANCE);
	}

	private static @Nullable SnapshotNode toSnapshotNode(final @Nullable Object value, final Class<?> type,
			final SnapshotNodeFactory factory)
	{
		final Class<?> specificType = value == null ? type : value.getClass();

		if (value == null || !factory.supports(specificType)) {
			return null;
		}

		return factory.toSnapshotNode(value);
	}
}
