package de.serra.so_dirty.sn;

import de.serra.so_dirty.difference.DifferenceNode;
import de.serra.so_dirty.difference.DifferenceNodes;
import de.serra.so_dirty.difference.DifferenceType;
import de.serra.so_dirty.difference.LeafDifferenceNode;

import org.checkerframework.checker.nullness.qual.Nullable;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.Objects;

import static de.serra.so_dirty.difference.DifferenceType.EQUALITY;
import static de.serra.so_dirty.difference.DifferenceType.REFERENCE;
import static de.serra.so_dirty.difference.DifferenceType.TYPE_CHANGE;

/**
 * Snapshot of an array and its members.
 *
 * @author Peter Lamby
 */
public class ArraySnapshotNode implements SnapshotNode {
	private final Object value;
	private final @Nullable SnapshotNode[] nodes;

	/**
	 * Constucts
	 *
	 * @param value   The instance of the array.
	 * @param members an array of nodes for every member of the array. The array must contain {@code null} if the member
	 *                at that index itself is null.
	 */
	public ArraySnapshotNode(final Object value, final @Nullable SnapshotNode[] members) {
		this.value = value;
		this.nodes = Arrays.copyOf(members, members.length);
	}

	@Override
	public Object value() {
		return value;
	}

	@Override
	public DifferenceNode diff(final @Nullable SnapshotNode other, final String path) {
		final var diffTypes = EnumSet.noneOf(DifferenceType.class);
		if (other == null || !Objects.equals(value, other.value())) {
			diffTypes.add(EQUALITY);
		}
		if (other == null || value != other.value()) {
			diffTypes.add(REFERENCE);
		}
		if (other == null || !Objects.equals(value.getClass(), other.value().getClass())) {
			diffTypes.add(TYPE_CHANGE);
		}

		if (!(other instanceof ArraySnapshotNode)) {
			// Other is not an array anymore
			return new LeafDifferenceNode(path, diffTypes.toArray(DifferenceType[]::new));
		}

		final ArraySnapshotNode them = (ArraySnapshotNode) other;
		final var ret = new DifferenceNodes(path, diffTypes.toArray(DifferenceType[]::new));

		// in case the arrays are of different length we only compare the members both have in common.
		// We already know for sure that members higher than min must have changed since they are either
		// new or not present anymore.
		//
		// This may change if we start to support order independent compares in the future.
		final int min = Math.min(nodes.length, them.nodes.length);
		if (nodes.length != them.nodes.length) {
			// since the arrays are of different length all values in the longer array are
			// automatically a difference.
			final int max = Math.max(nodes.length, them.nodes.length);
			for (int i = min; i < max; i++) {
				// NOTE(perf) we can make this more efficient by just recording the changed indices.
				ret.add(new LeafDifferenceNode("[" + i + "]", TYPE_CHANGE, REFERENCE));
			}
		}

		// an empty array can't have different members
		if (min > 0) {
			// check every member
			for (int i = 0; i < min; i++) {
				final var snapshotNode = nodes[i];
				final var otherSnapshotNode = them.nodes[i];
				// both members stayed null
				if (snapshotNode == null && otherSnapshotNode == null) {
					continue;
				}

				final var diff = SnapshotNode.doDiff(snapshotNode, otherSnapshotNode, "[" + i + "]");
				if (diff.isDifferentRecursive((DifferenceType[]) null)) {
					// only record actual differences
					ret.add(diff);
				}
			}
		}

		return ret;
	}
}
