package de.serra.so_dirty.sn;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import org.checkerframework.checker.nullness.qual.Nullable;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.HashMap;

/**
 * Factory that creates an {@link ClassSnapshotNode}.
 *
 * @author Peter Lamby
 */
public class ClassSnapshotNodeFactory implements SnapshotNodeFactory {
	private final SnapshotNodeFactory memberFactory;

	/**
	 * Constructs.
	 * <p>
	 * Uses {@code memberFactory} to create {@link SnapshotNode}s for the fields of the class.
	 *
	 * @param memberFactory The factory to use for the fields of the class.
	 */
	public ClassSnapshotNodeFactory(final SnapshotNodeFactory memberFactory) {
		this.memberFactory = memberFactory;
	}

	@Override
	public boolean supports(final Class<?> type) {
		return !(type.isPrimitive() || type.isEnum() || type.isArray() /* || type.isRecord() */
				|| type.isSynthetic() || type.isInterface() || type.isAnnotation());
	}

	@Override
	public ClassSnapshotNode toSnapshotNode(final Object value) {
		final var fields = value.getClass().getDeclaredFields();
		final HashMap<String, @Nullable SnapshotNode> fieldsToNodes = new HashMap<>(fields.length);
		for (final var field : fields) {
			// NOTE make configurable
			if (field.isSynthetic() || Modifier.isStatic(field.getModifiers())) {
				continue;
			}

			// NOTE make this behavior configurable. skip, tryMakeAcessible or error
			if (!field.canAccess(value)) {
				if (!field.trySetAccessible()) {
					continue;
				}
			}

			final Object fieldValue = getFieldValue(field, value);
			if (fieldValue != null && memberFactory.supports(fieldValue.getClass())) {
				fieldsToNodes.put(field.getName(), memberFactory.toSnapshotNode(fieldValue));
			}
		}
		return new ClassSnapshotNode(value, fieldsToNodes);
	}

	@SuppressFBWarnings(
			value = "EXS_EXCEPTION_SOFTENING_NO_CONSTRAINTS",
			justification = "The exception should never happen since we check canAccess and trySetAccessible first")
	private @Nullable Object getFieldValue(final Field field, final Object value) {
		try {
			return field.get(value);
		} catch (IllegalArgumentException | IllegalAccessException e) {
			throw new IllegalStateException("Can not access field that should be accessible according to our checks", e);
		}

	}
}
