package de.serra.so_dirty.sn;

import org.checkerframework.checker.nullness.qual.Nullable;

import java.util.HashMap;
import java.util.Map;

/**
 * Always returns {@code null} when {@link SnapshotNodeFactory#toSnapshotNode(Object)} is called.
 * <p>
 * Used to exclude some types that should not included in Snapshots.
 * <p>
 * Add this factory first to {@link de.serra.so_dirty.sn.CompoundSnapshotNodeFactory#add(SnapshotNodeFactory)
 * CompoundSnapshotNodeFactory.add(SnapshotNodeFactory)} to make sure no other factory sees the value.
 *
 * @author Peter Lamby
 */
public class IgnoreSnapshotNodeFactory implements SnapshotNodeFactory {
	/**
	 * If a {@link Class} is in this map it is ignored. Depending on the {@link java.util.Map.Entry#getValue() value}
	 * subtypes should also be ignored.
	 */
	private final Map<Class<?>, Boolean> entries = new HashMap<>();

	/**
	 * Constructs.
	 */
	public IgnoreSnapshotNodeFactory() {
	}

	/**
	 * Adds a type that should be ignored by this Factory.
	 *
	 * @param type            The type to ignore.
	 * @param includeSubtypes whether to also ignore subtypes of {@code type}.
	 */
	public void add(final Class<?> type, final boolean includeSubtypes) {
		entries.put(type, includeSubtypes);
	}

	@Override
	public boolean supports(final Class<?> type) {
		if (entries.containsKey(type)) {
			return true;
		}

		@Nullable
		Class<?> superType = type;
		while ((superType = superType.getSuperclass()) != null) {
			if (Boolean.TRUE.equals(entries.get(superType))) {
				return true;
			}
		}

		return false;
	}

	@Override
	public @Nullable SnapshotNode toSnapshotNode(final Object value) {
		return null;
	}
}
