package de.serra.so_dirty.util;

import org.checkerframework.checker.nullness.qual.Nullable;

import java.util.Objects;

/**
 * Helpers to work with path expressions.
 *
 * @author Peter Lamby
 */
public final class PathUtil {
	/**
	 * Constructs.
	 *
	 * Private because this is a utility class with only static helpers.
	 */
	private PathUtil() {
	}

	/**
	 * Returns the last part of the path expression.
	 *
	 * @param pathExpression the expression.
	 * @return The last part.
	 */
	public static String getLastPart(final String pathExpression) {
		if (pathExpression.isEmpty()) {
			return pathExpression;
		}

		final var chars = pathExpression.toCharArray();
		final var buf = new StringBuilder();

		// search for a . or ] from the end.
		for (int i = chars.length - 1; i >= 0; i--) {
			var c = chars[i];

			if (c == '.') {
				return buf.length() == 0 ? "." : buf.reverse().toString();
			} else if (c == ']') {
				if (buf.length() != 0) {
					return buf.reverse().toString();
				} else {
					// search for the start of the [] pair and return the index expression.
					var found = false;
					for (i--; i >= 0; i--) {
						c = chars[i];
						if (c == '[') {
							found = true;
							break;
						}
						buf.append(c);
					}
					if (!found) {
						throw new IllegalArgumentException("Found ']' without a matching '['.");
					}
					if (buf.length() == 0) {
						throw new IllegalArgumentException("Found an empty index expression.");
					}
					return "[" + buf.reverse() + "]";
				}
			} else if (c == '[') {
				throw new IllegalArgumentException("Found '[' without a matching ']'.");
			} else {
				buf.append(c);
			}
		}

		return buf.reverse().toString();
	}

	/**
	 * Returns the first part of the path expression.
	 *
	 * @param pathExpression the expression.
	 * @return The first part.
	 */
	public static FirstPart getFirstPart(final String pathExpression) {
		if (pathExpression.isEmpty()) {
			return new FirstPart(pathExpression);
		}

		final var chars = pathExpression.toCharArray();
		final var buf = new StringBuilder();

		for (int i = 0; i < chars.length; i++) {
			var c = chars[i];
			if (c == '.') {
				if (buf.length() == 0) {
					return new FirstPart(".", getTail(chars, ++i /* Only the part after the . */));
				} else {
					return new FirstPart(buf.toString(), getTail(chars, ++i /* Only the part after the . */));
				}
			} else if (c == '[') {
				if (buf.length() != 0) {
					return new FirstPart(buf.toString(), getTail(chars, i /* include the [ */));
				} else {
					// search for the end of the [] pair and return the index expression.
					var found = false;
					for (i++; i < chars.length; i++) {
						c = chars[i];
						if (c == ']') {
							found = true;
							break;
						}
						buf.append(c);
					}
					if (!found) {
						throw new IllegalArgumentException("Found '[' without a matching ']'.");
					}
					if (buf.length() == 0) {
						throw new IllegalArgumentException("Found an empty index expression.");
					}
					if (i + 1 < chars.length && chars[i + 1] == '.') {
						/* Only the part after the ]. */
						i++;
					}
					return new FirstPart("[" + buf + "]",
							getTail(chars, ++i/* Only the part after the ']' and optionally the '.' (if present) */));
				}
			} else if (c == ']') {
				throw new IllegalArgumentException("Found ']' without a matching '['.");
			} else {
				buf.append(c);
			}
		}

		return new FirstPart(buf.toString());
	}

	/**
	 * Returns a String made up of the {@code chars} path expression beginning at {@code offset}
	 *
	 * <p>
	 * Will return {@code null} if the tail would be empty or after is out of bounds.
	 *
	 * @param chars  The chars to create a tail from.
	 * @param offset The index where the tail of the expression should start.
	 * @return The tail if any or {@code null}.
	 */
	public static @Nullable String getTail(final char[] chars, final int offset) {
		if (offset >= chars.length) {
			// would be out of bounds or empty
			return null;
		}

		return new String(chars, offset, chars.length - offset);
	}

	/**
	 * Contains the first part of an path expression and it's tail.
	 *
	 * @author Peter Lamby
	 */
	public static class FirstPart {
		/**
		 * The first part.
		 */
		public final String firstPart;
		/**
		 * The tail.
		 */
		public final @Nullable String rest;

		public FirstPart(final String firstPart, final @Nullable String rest) {
			this.firstPart = firstPart;
			this.rest = rest;
		}

		public FirstPart(final String firstPart) {
			this(firstPart, null);
		}

		@Override
		public int hashCode() {
			return Objects.hash(firstPart, rest);
		}

		@Override
		public boolean equals(final @Nullable Object obj) {
			if (this == obj) {
				return true;
			}
			if (obj == null) {
				return false;
			}
			if (getClass() != obj.getClass()) {
				return false;
			}
			final FirstPart other = (FirstPart) obj;
			return Objects.equals(firstPart, other.firstPart) && Objects.equals(rest, other.rest);
		}
	}
}
